--
-- CREATE TABLE: Site
--
CREATE TABLE Site
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	Type VARCHAR(32),
	Description VARCHAR(256),
	PRIMARY KEY (ID),
	UNIQUE (Name)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON Site
(
	Name
);

--
-- CREATE INDEX: SiteType
--
CREATE INDEX SiteType ON Site
(
	Type
);
--
-- CREATE TABLE: Hardware
--
CREATE TABLE Hardware
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Site INTEGER NOT NULL,
	Memory INTEGER,
	DiskSpace INTEGER,
	CPU INTEGER,
	Description VARCHAR(256),
	Name VARCHAR(64) NOT NULL,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON Hardware
(
	Site,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Hardware ADD 
	FOREIGN KEY (Site)
		REFERENCES Site (ID);
		
--
-- CREATE TABLE: Platform
--
CREATE TABLE Platform
(
	ID INTEGER,
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	PathCaseSensitive BIT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON Platform
(
	Name
);

--
-- CREATE TABLE: OpSystem
--
CREATE TABLE OpSystem
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Hardware INTEGER,
	Name VARCHAR(64) NOT NULL,
	HostName VARCHAR(64) NOT NULL,
	Type VARCHAR(64),
	Edition VARCHAR(256),
	Description VARCHAR(256),
	Memory INTEGER,
	DiskSpace INTEGER,
	CPU INTEGER,
	Platform INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Host
--
CREATE UNIQUE INDEX Host ON OpSystem
(
	HostName
);

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON OpSystem
(
	Hardware,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE OpSystem ADD 
	FOREIGN KEY (Hardware)
		REFERENCES Hardware (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE OpSystem ADD 
	FOREIGN KEY (Platform)
		REFERENCES Platform (ID);

--
-- CREATE TABLE: EngineStatus
--
CREATE TABLE EngineStatus
(
	Status VARCHAR(32),
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: Volume
--
CREATE TABLE Volume
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	OpSystem INTEGER NOT NULL,
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(256),
	TotalDiskSpace INTEGER,
	AvailDiskSpace INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON Volume
(
	OpSystem,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Volume ADD 
	FOREIGN KEY (OpSystem)
		REFERENCES OpSystem (ID);

--
-- CREATE TABLE: DbEngine
--
CREATE TABLE DbEngine
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	LogicalName VARCHAR(64),
	OpSystem INTEGER NOT NULL,
	WorkingVolume INTEGER,
	WorkingDir VARCHAR(512),
	StatusTime TIMESTAMP,
	LastCheckTime TIMESTAMP,
	Name VARCHAR(64) NOT NULL,
	EngineStatus SMALLINT,
	Description VARCHAR(256),
	AuthInfo VARBINARY(512),
	IsSQL BIT,
	PortNbr INTEGER,
	SQLPortNbr INTEGER,
	LocalVolume INTEGER,
	LocalDir VARCHAR(512),
	Users INTEGER,
	ReleaseVersion VARCHAR(32),
	Serial VARCHAR(32),
	LicenseOEM INTEGER,
	LicenseType VARCHAR(32),
	LicenseCPUNbr INTEGER,
	LicenseServerType VARCHAR(32),
	LicenseUserNbr INTEGER,
	IsAgent BIT,
	Type VARCHAR(32),
	ReplNodeID VARCHAR(32),
	IPAddress VARCHAR(39),
	AcceptConn BIT,
	ReadOnly INTEGER,
	WebServerConfig VARCHAR(1024),
	ConfigFilePath VARCHAR(512),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: LocalDir
--
CREATE INDEX LocalDir ON DbEngine
(
	OpSystem,
	LocalVolume,
	LocalDir,
	EngineStatus
);

--
-- CREATE INDEX: Logical_Name
--
CREATE INDEX Logical_Name ON DbEngine
(
	LogicalName
);

--
-- CREATE INDEX: Name
--
CREATE INDEX Name ON DbEngine
(
	OpSystem,
	Name
);

--
-- CREATE INDEX: WorkingDir
--
CREATE UNIQUE INDEX WorkingDir ON DbEngine
(
	OpSystem,
	WorkingVolume,
	WorkingDir
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE DbEngine ADD 
	FOREIGN KEY (EngineStatus)
		REFERENCES EngineStatus (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE DbEngine ADD 
	FOREIGN KEY (OpSystem)
		REFERENCES OpSystem (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE DbEngine ADD 
	FOREIGN KEY (WorkingVolume)
		REFERENCES Volume (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_4
--
ALTER TABLE DbEngine ADD 
	FOREIGN KEY (LocalVolume)
		REFERENCES Volume (ID);

--
-- CREATE TABLE: ActionType
--
CREATE TABLE ActionType
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Description VARCHAR(64),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: Agent
--
CREATE TABLE Agent
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	OpSystem INTEGER,
	WorkingVolume INTEGER,
	WorkingDir VARCHAR(512),
	DbEngine INTEGER,
	Status SMALLINT,
	StatusTime TIMESTAMP,
	PRIMARY KEY (ID),
	UNIQUE (DbEngine)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Path
--
CREATE UNIQUE INDEX Path ON Agent
(
	WorkingDir,
	WorkingVolume
);

--
-- CREATE INDEX: DBEngine
--
CREATE UNIQUE INDEX DBEngine ON Agent
(
	DbEngine
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Agent ADD 
	FOREIGN KEY (OpSystem)
		REFERENCES OpSystem (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE Agent ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE Agent ADD 
	FOREIGN KEY (Status)
		REFERENCES EngineStatus (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_4
--
ALTER TABLE Agent ADD 
	FOREIGN KEY (WorkingVolume)
		REFERENCES Volume (ID);

--
-- CREATE TABLE: Consumer
--
CREATE TABLE Consumer
(
	Name VARCHAR(32) NOT NULL,
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Lib VARCHAR(32),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: ActiveConsumer
--
CREATE TABLE AgentConsumer
(
	Agent INTEGER,
	Consumer SMALLINT,
	ThreadCount INTEGER,
	IsActive BIT,
	StatusTime TIMESTAMP
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: AgentConsumer
--
CREATE UNIQUE INDEX AgentConsumer ON AgentConsumer
(
	Agent,
	Consumer
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_0
--
ALTER TABLE AgentConsumer ADD 
	FOREIGN KEY (Agent)
		REFERENCES Agent (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE AgentConsumer ADD 
	FOREIGN KEY (Consumer)
		REFERENCES Consumer (ID);

--
-- CREATE TABLE: DatabaseTable
--
CREATE TABLE DatabaseTable
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	DBEngine INTEGER NOT NULL,
	Name VARCHAR(64) NOT NULL,
	DictFile VARCHAR(512),
	isSQL BIT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON DatabaseTable
(
	DBEngine,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE DatabaseTable ADD 
	FOREIGN KEY (DBEngine)
		REFERENCES DbEngine (ID);

--
-- CREATE TABLE: EntityType
--
CREATE TABLE EntityType
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: Action
--
CREATE TABLE Action
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	OpSystem INTEGER NOT NULL,
	Agent INTEGER,
	ReplicationPlan INTEGER,
	Consumer SMALLINT,
	TimeProcessed TIMESTAMP,
	Description VARCHAR(1024),
	Parameters VARCHAR(1024),
	ErrorCode INTEGER,
	ErrorMsg VARCHAR(1024),
	Displayed BIT,
	ActionType SMALLINT NOT NULL,
	EntityType SMALLINT NOT NULL,
	EntityPK INTEGER NOT NULL,
	ActionTime TIMESTAMP NOT NULL,
	Progress TINYINT NOT NULL,
	ParentActionID INTEGER,
	FileVolume INTEGER,
	FilePath VARCHAR(512),
	FileName VARCHAR(260),
	OwnerActionID INTEGER,
	OwnerActionFactor DOUBLE,
	Content LVARBINARY,
	ContentLen INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Entity
--
CREATE INDEX Entity ON Action
(
	TimeProcessed,
	OpSystem,
	EntityType,
	EntityPK,
	FileVolume,
	FilePath,
	FileName
);

--
-- CREATE INDEX: Error_Code
--
CREATE INDEX Error_Code ON Action
(
	ErrorCode,
	Displayed
);

--
-- CREATE INDEX: Owner
--
CREATE INDEX Owner ON Action
(
	OwnerActionID
);

--
-- CREATE INDEX: Unprocessed
--
CREATE INDEX Unprocessed ON Action
(
	OpSystem,
	Consumer,
	TimeProcessed,
	ID,
	Progress
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Action ADD 
	FOREIGN KEY (ActionType)
		REFERENCES ActionType (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE Action ADD 
	FOREIGN KEY (EntityType)
		REFERENCES EntityType (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE Action ADD 
	FOREIGN KEY (Agent)
		REFERENCES Agent (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_4
--
ALTER TABLE Action ADD 
	FOREIGN KEY (Consumer)
		REFERENCES Consumer (ID);

		
--
-- CREATE TABLE: ActionHistory
--
CREATE TABLE ActionHistory
(
	ID INTEGER,
	GUID BINARY(16),
	OpSystem INTEGER NOT NULL,
	Agent INTEGER,
	ReplicationPlan INTEGER,
	Consumer SMALLINT,
	TimeProcessed TIMESTAMP,
	Description VARCHAR(1024),
	Parameters VARCHAR(1024),
	ErrorCode INTEGER,
	ErrorMsg VARCHAR(1024),
	Displayed BIT,
	ActionType SMALLINT NOT NULL,
	EntityType SMALLINT NOT NULL,
	EntityPK INTEGER NOT NULL,
	ActionTime TIMESTAMP NOT NULL,
	Progress TINYINT NOT NULL,
	ParentActionID INTEGER,
	FileVolume INTEGER,
	FilePath VARCHAR(512),
	FileName VARCHAR(260),
	OwnerActionID INTEGER,
	OwnerActionFactor DOUBLE,
	Content LVARBINARY,
	ContentLen INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE INDEX: Entity
--
CREATE INDEX Entity ON ActionHistory
(
	TimeProcessed,
	OpSystem,
	EntityType,
	EntityPK,
	FileVolume,
	FilePath,
	FileName
);

--
-- CREATE INDEX: Error_Code
--
CREATE INDEX Error_Code ON ActionHistory
(
	ErrorCode,
	Displayed
);

--
-- CREATE INDEX: Owner
--
CREATE INDEX Owner ON ActionHistory
(
	OwnerActionID
);

--
-- CREATE TABLE: FileType
--
CREATE TABLE FileType
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(256),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';


--
-- CREATE TABLE: Publication
--
CREATE TABLE Publication
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	DbEngine INTEGER NOT NULL,
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(256),
	Type VARCHAR(32),
	PublicationMode SMALLINT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON Publication
(
	DbEngine,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Publication ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);


--
-- CREATE TABLE: ReplicationStatus
--
CREATE TABLE ReplicationStatus
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Name VARCHAR(20),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: DbEngineGroup
--
CREATE TABLE DbEngineGroup
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	Type VARCHAR(32),
	Description VARCHAR(256),
	PRIMARY KEY (ID),
	UNIQUE (Name)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: FileGroup
--
CREATE TABLE FileGroup
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	DbEngine INTEGER NOT NULL,
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(256),
	Type VARCHAR(32),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON FileGroup
(
	DbEngine,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE FileGroup ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);

--
-- CREATE TABLE: FileTable
--
CREATE TABLE FileTable
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Volume INTEGER NOT NULL,
	Path VARCHAR(512),
	Name VARCHAR(260) NOT NULL,
	DateCreated TIMESTAMP,
	DateUpdated TIMESTAMP,
	Owner VARCHAR(64),
	FileSize BIGINT,
	FileType SMALLINT,
	InUse BIT,
	DatabaseID INTEGER,
	TableName VARCHAR(64),
	IsPartitioned BIT,
	IsRemoved BIT,
	Description VARCHAR(512),
	ErrorCd INTEGER,
	SysErrorCd INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: DatabaseInd
--
CREATE INDEX DatabaseInd ON FileTable
(
	DatabaseID
);

--
-- CREATE INDEX: InUse
--
CREATE INDEX InUse ON FileTable
(
	Volume,
	InUse
);

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON FileTable
(
	Volume,
	Path,
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE FileTable ADD 
	FOREIGN KEY (Volume)
		REFERENCES Volume (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE FileTable ADD 
	FOREIGN KEY (FileType)
		REFERENCES FileType (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE FileTable ADD 
	FOREIGN KEY (DatabaseID)
		REFERENCES DatabaseTable (ID);

--
-- CREATE TABLE: Layout
--
CREATE TABLE Layout
(
	layoutIdx SMALLINT NOT NULL,
	GUID BINARY(16),
	description VARCHAR(64) NOT NULL,
	zoom SMALLINT NOT NULL,
	panx SMALLINT NOT NULL,
	pany SMALLINT NOT NULL,
	ID INTEGER IDENTITY (0,1),
	PRIMARY KEY (layoutIdx),
	UNIQUE (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: LogAgent
--
CREATE TABLE LogAgent
(
	ID SMALLINT NOT NULL,
	Name VARCHAR(64),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: LogType
--
CREATE TABLE LogType
(
	ID SMALLINT NOT NULL,
	Name VARCHAR(64) NOT NULL,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: ReplicationPlan
--
CREATE TABLE ReplicationPlan
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Name VARCHAR(32) NOT NULL,
	Description VARCHAR(1024),
	SourceDBEngine INTEGER NOT NULL,
	TargetDBEngine INTEGER NOT NULL,
	IsSync BIT,
	WarningThreshold INTEGER,
	ErrorThreshold INTEGER,
	Latency INTEGER,
	LatencyTime TIMESTAMP,
	LatencyRev INTEGER,
	LatencyRevTime TIMESTAMP,
	FileMask VARCHAR(64),
	ReadTimeout INTEGER,
	BatchSize INTEGER,
	LockRetrySleep INTEGER,
	DebugMinlog BIT,
	RememberLogPos BIT,
	ParallelApply BIT, 
	CheckDependencies BIT, 
	SyncLogWrites BIT, 
	AnalyzerThreadsCount INTEGER, 
	ApplyThreadsCount INTEGER,
	ExceptionMode INTEGER,
	LogChangeDetails VARCHAR(512),
	LockRetryCount INTEGER,
	IsDeployed BIT,
	IsPulling BIT,
	IgnorePartition BIT,
	ExtensionLibrary VARCHAR(1024),
	RebuildOnDeploy BIT,
	SourceLogPos VARCHAR(32),
	SourceTranID VARCHAR(32),
	TargetLogPos VARCHAR(32),
	TargetTranID VARCHAR(32),
	Status SMALLINT NOT NULL,
	StatusTime TIMESTAMP,
	StatusMsg VARCHAR(1024),
	StatusActionID INTEGER,
	SourceAgentID INTEGER,
	TargetAgentID INTEGER,
	IsMonitorActive BIT,
	MonitorTime TIMESTAMP,
	RefreshRate INTEGER,
	IsExcepLogActive BIT,
	ExcepLogTime TIMESTAMP,
	IsFuncTimeActive BIT,
	FuncTimeTime TIMESTAMP,
	IsPaused BIT,
	PauseTime TIMESTAMP,
	ConflictDetection BIT,
	FileFilter VARCHAR(128),
	LastExcepAck TIMESTAMP,
	HasNonAckException BIT,
	IsInExceptionDump BIT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE UNIQUE INDEX Name ON ReplicationPlan
(
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplicationPlan ADD 
	FOREIGN KEY (SourceDBEngine)
		REFERENCES DbEngine (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE ReplicationPlan ADD 
	FOREIGN KEY (TargetDBEngine)
		REFERENCES DbEngine (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE ReplicationPlan ADD 
	FOREIGN KEY (Status)
		REFERENCES ReplicationStatus (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_5
--
ALTER TABLE ReplicationPlan ADD 
	FOREIGN KEY (SourceAgentID)
		REFERENCES Agent (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_6
--
ALTER TABLE ReplicationPlan ADD 
	FOREIGN KEY (TargetAgentID)
		REFERENCES Agent (ID);

--
-- CREATE TABLE: Subscription
--
CREATE TABLE Subscription
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(1024),
	Publication INTEGER NOT NULL,
	DbEngine INTEGER NOT NULL,
	BiDirectional BIT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE INDEX Name ON Subscription
(
	Name
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Subscription ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE Subscription ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);


--
-- CREATE TABLE: DbEngineGroupItem
--
CREATE TABLE DbEngineGroupItem
(
	DbEngineGroup INTEGER NOT NULL,
	DbEngine INTEGER NOT NULL,
	PRIMARY KEY (DbEngineGroup, DbEngine)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE DbEngineGroupItem ADD 
	FOREIGN KEY (DbEngineGroup)
		REFERENCES DbEngineGroup (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE DbEngineGroupItem ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);

--
-- CREATE TABLE: FileGroupItem
--
CREATE TABLE FileGroupItem
(
	FileGroup INTEGER NOT NULL,
	FileTable INTEGER NOT NULL,
	PRIMARY KEY (FileGroup, FileTable)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE FileGroupItem ADD 
	FOREIGN KEY (FileGroup)
		REFERENCES FileGroup (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE FileGroupItem ADD 
	FOREIGN KEY (FileTable)
		REFERENCES FileTable (ID);

--
-- CREATE TABLE: Globals
--
CREATE TABLE Globals
(
	KeyName VARCHAR(64) NOT NULL,
	KeyValue VARCHAR(1024),
	PRIMARY KEY (KeyName)
) STORAGE_ATTRIBUTES 'hotalter';
--
-- CREATE TABLE: IP
--
CREATE TABLE IP
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	OpSystem INTEGER,
	Address VARCHAR(39),
	Type VARCHAR(32),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Address
--
CREATE INDEX Address ON IP
(
	Address
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE IP ADD 
	FOREIGN KEY (OpSystem)
		REFERENCES OpSystem (ID);

--
-- CREATE TABLE: LayoutItem
--
CREATE TABLE LayoutItem
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	layout SMALLINT,
	objectType SMALLINT NOT NULL,
	objectId SMALLINT NOT NULL,
	posX SMALLINT,
	posY SMALLINT,
	height SMALLINT,
	width SMALLINT,
	collapsed BIT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: layout_group
--
CREATE UNIQUE INDEX layout_group ON LayoutItem
(
	layout,
	objectType,
	objectId
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE LayoutItem ADD 
	FOREIGN KEY (layout)
		REFERENCES Layout (layoutIdx);

--
-- CREATE TABLE: LogTable
--
CREATE TABLE LogTable
(
	LogTime TIMESTAMP NOT NULL,
	LogType SMALLINT NOT NULL,
	LogAgent SMALLINT NOT NULL,
	DBEngine INTEGER,
	AgentName VARCHAR(64),
	FunctionName VARCHAR(64) NOT NULL,
	DetailLevel SMALLINT NOT NULL,
	SeverityLevel SMALLINT NOT NULL,
	ErrorCode INTEGER NOT NULL,
	Message VARCHAR(1024) NOT NULL
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Index_1
--
CREATE INDEX Index_1 ON LogTable
(
	LogTime
);

--
-- CREATE INDEX: Index_2
--
CREATE INDEX Index_2 ON LogTable
(
	DBEngine
);

--
-- CREATE INDEX: Index_3
--
CREATE INDEX Index_3 ON LogTable
(
	LogAgent
);

--
-- CREATE INDEX: Index_4
--
CREATE INDEX Index_4 ON LogTable
(
	FunctionName
);

--
-- CREATE INDEX: Index_5
--
CREATE INDEX Index_5 ON LogTable
(
	SeverityLevel
);

--
-- CREATE INDEX: Index_6
--
CREATE INDEX Index_6 ON LogTable
(
	ErrorCode
);

--
-- CREATE INDEX: Index_7
--
CREATE INDEX Index_7 ON LogTable
(
	DetailLevel
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE LogTable ADD 
	FOREIGN KEY (LogType)
		REFERENCES LogType (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE LogTable ADD 
	FOREIGN KEY (LogAgent)
		REFERENCES LogAgent (ID);
		
--
-- CREATE TABLE: LogTableHistory
--
CREATE TABLE LogTableHistory
(
	LogTime TIMESTAMP NOT NULL,
	LogType SMALLINT NOT NULL,
	LogAgent SMALLINT NOT NULL,
	DBEngine INTEGER,
	AgentName VARCHAR(64),
	FunctionName VARCHAR(64) NOT NULL,
	DetailLevel SMALLINT NOT NULL,
	SeverityLevel SMALLINT NOT NULL,
	ErrorCode INTEGER NOT NULL,
	Message VARCHAR(1024) NOT NULL
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Index_1
--
CREATE INDEX Index_1 ON LogTableHistory
(
	LogTime
);

--
-- CREATE INDEX: Index_2
--
CREATE INDEX Index_2 ON LogTableHistory
(
	DBEngine
);

--
-- CREATE INDEX: Index_3
--
CREATE INDEX Index_3 ON LogTableHistory
(
	LogAgent
);

--
-- CREATE INDEX: Index_4
--
CREATE INDEX Index_4 ON LogTableHistory
(
	FunctionName
);

--
-- CREATE INDEX: Index_5
--
CREATE INDEX Index_5 ON LogTableHistory
(
	SeverityLevel
);

--
-- CREATE INDEX: Index_6
--
CREATE INDEX Index_6 ON LogTableHistory
(
	ErrorCode
);

--
-- CREATE INDEX: Index_7
--
CREATE INDEX Index_7 ON LogTableHistory
(
	DetailLevel
);

--
-- CREATE TABLE: Mac
--
CREATE TABLE Mac
(
	Address VARCHAR(32) NOT NULL,
	GUID BINARY(16),
	Hardware INTEGER NOT NULL,
	Type VARCHAR(32),
	PRIMARY KEY (Address)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Mac ADD 
	FOREIGN KEY (Hardware)
		REFERENCES Hardware (ID);

--
-- CREATE TABLE: PublicationFile
--
CREATE TABLE PublicationFile
(
	Publication INTEGER NOT NULL,
	FileTable INTEGER NOT NULL,
	PRIMARY KEY (Publication, FileTable)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationFile ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE PublicationFile ADD 
	FOREIGN KEY (FileTable)
		REFERENCES FileTable (ID);

--
-- CREATE TABLE: PublicationFolder
--
CREATE TABLE PublicationFolder
(
	Publication INTEGER NOT NULL,
	Folder INTEGER NOT NULL,
	ListType SMALLINT NOT NULL,
	PRIMARY KEY (Publication, Folder)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationFolder ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE PublicationFolder ADD 
	FOREIGN KEY (Folder)
		REFERENCES FileTable (ID);
		
--
-- CREATE TABLE: PublicationFolderRules
--
CREATE TABLE PublicationFolderRules
(
	Publication INTEGER NOT NULL,
	Folder INTEGER NOT NULL,
	Type SMALLINT NOT NULL,
	Rule VARCHAR(512) NOT NULL
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationFolderRules ADD 
	FOREIGN KEY (Publication, Folder)
		REFERENCES PublicationFolder (Publication, Folder);

--
-- CREATE INDEX: FolderRules
--
CREATE INDEX FolderRules ON PublicationFolderRules
(
	Publication,
	Folder,
	Type
);


--
-- CREATE TABLE: PublicationDatabase
--
CREATE TABLE PublicationDatabase
(
	Publication INTEGER NOT NULL,
	DatabaseID INTEGER NOT NULL,
	ListType SMALLINT NOT NULL,
	PRIMARY KEY (Publication, DatabaseID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationDatabase ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE PublicationDatabase ADD 
	FOREIGN KEY (DatabaseID)
		REFERENCES DatabaseTable (ID);
		
--
-- CREATE TABLE: PublicationDatabaseRules
--
CREATE TABLE PublicationDatabaseRules
(
	Publication INTEGER NOT NULL,
	DatabaseID INTEGER NOT NULL,
	Type SMALLINT NOT NULL,
	Rule VARCHAR(512) NOT NULL
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationDatabaseRules ADD 
	FOREIGN KEY (Publication, DatabaseID)
		REFERENCES PublicationDatabase (Publication, DatabaseID);

--
-- CREATE INDEX: DatabaseRules
--
CREATE INDEX DatabaseRules ON PublicationDatabaseRules
(
	Publication,
	DatabaseID,
	Type
);


--
-- CREATE TABLE: PublicationDictionary
--
CREATE TABLE PublicationDictionary
(
	Publication INTEGER NOT NULL,
	DatabaseID INTEGER NOT NULL,
	DictionaryFileID INTEGER NOT NULL,
	PRIMARY KEY (Publication, DatabaseID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationDictionary ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE PublicationDictionary ADD 
	FOREIGN KEY (DatabaseID)
		REFERENCES DatabaseTable (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE PublicationDictionary ADD 
	FOREIGN KEY (DictionaryFileID)
		REFERENCES FileTable (ID);
		
		
--
-- CREATE TABLE: PublicationFileGroup
--
CREATE TABLE PublicationFileGroup
(
	Publication INTEGER NOT NULL,
	FileGroup INTEGER NOT NULL,
	PRIMARY KEY (Publication, FileGroup)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE PublicationFileGroup ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE PublicationFileGroup ADD 
	FOREIGN KEY (FileGroup)
		REFERENCES FileGroup (ID);

--
-- CREATE TABLE: Redirection
--
CREATE TABLE Redirection
(
	Subscription INTEGER NOT NULL,
	SourcePath VARCHAR(512) NOT NULL,
	TargetPath VARCHAR(512) NOT NULL,
	Reverse BIT,
	PRIMARY KEY (Subscription, SourcePath)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE Redirection ADD 
	FOREIGN KEY (Subscription)
		REFERENCES Subscription (ID);

--
-- CREATE TABLE: ReplAgent
--
CREATE TABLE ReplAgent
(
	Action INTEGER NOT NULL,
	OpSystem INTEGER NOT NULL,
	Path VARCHAR(512) NOT NULL,
	Name VARCHAR(64) NOT NULL,
	SourceServer VARCHAR(128) NOT NULL,
	TargetServer VARCHAR(128) NOT NULL,
	SourceNodeID VARCHAR(64),
	TargetNodeID VARCHAR(64),
	ErrorCode INTEGER,
	Message VARCHAR(1024),
	PRIMARY KEY (Action, OpSystem, Path)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: ReplExcep
--
CREATE TABLE ReplExcep
(
	ReplicationPlan INTEGER NOT NULL,
	ExceptionTime TIMESTAMP NOT NULL,
	SequenceNumber BIGINT IDENTITY (0,1),
	GUID BINARY(16),
	Direction SMALLINT,
	TransactionNumber BIGINT,
	LogEntryTime TIMESTAMP,
	LogNumber INTEGER,
	LogPosition INTEGER,
	FileID INTEGER,
	Operation VARCHAR(32),
	ErrorCode INTEGER,
	FileName VARCHAR(1024),
	PRIMARY KEY (ReplicationPlan, ExceptionTime, SequenceNumber)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: ReplExcepHistory
--
CREATE TABLE ReplExcepHistory
(
	ReplicationPlan INTEGER NOT NULL,
	ExceptionTime TIMESTAMP NOT NULL,
	SequenceNumber BIGINT,
	GUID BINARY(16),
	Direction SMALLINT,
	TransactionNumber BIGINT,
	LogEntryTime TIMESTAMP,
	LogNumber INTEGER,
	LogPosition INTEGER,
	FileID INTEGER,
	Operation VARCHAR(32),
	ErrorCode INTEGER,
	FileName VARCHAR(1024),
	PRIMARY KEY (ReplicationPlan, ExceptionTime, SequenceNumber)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplExcep ADD 
	FOREIGN KEY (ReplicationPlan)
		REFERENCES ReplicationPlan (ID);

--
-- CREATE INDEX: ExceptionTime
--
CREATE INDEX ExceptionTime ON ReplExcep
(
	ReplicationPlan,
	ExceptionTime
);

--
-- CREATE TABLE: ReplStat
--
CREATE TABLE ReplStat
(
	ID BIGINT IDENTITY (0,1),
	ReplicationPlan INTEGER NOT NULL,
	Direction SMALLINT NOT NULL,
	StatTime TIMESTAMP NOT NULL,
	SourceDBConnected BIT,
	TargetDBConnected BIT,
	LogNumber INTEGER,
	LogPosition INTEGER,
	State INTEGER,
	SequenceNumber BIGINT,
	FunctionName VARCHAR(64),
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT,
	Latency INTEGER,
	PRIMARY KEY (ReplicationPlan, Direction, StatTime)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplStat ADD 
	FOREIGN KEY (ReplicationPlan)
		REFERENCES ReplicationPlan (ID);
		
--
-- CREATE TABLE: ReplStatHistory
--
CREATE TABLE ReplStatHistory
(
	ID BIGINT,
	ReplicationPlan INTEGER NOT NULL,
	Direction SMALLINT NOT NULL,
	StatTime TIMESTAMP NOT NULL,
	SourceDBConnected BIT,
	TargetDBConnected BIT,
	LogNumber INTEGER,
	LogPosition INTEGER,
	State INTEGER,
	SequenceNumber BIGINT,
	FunctionName VARCHAR(64),
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT,
	Latency INTEGER,
	PRIMARY KEY (ReplicationPlan, Direction, StatTime)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: ReplParallelStat
--
CREATE TABLE ReplParallelStat
(
	ID BIGINT IDENTITY (0,1),
	ReplicationPlan INTEGER NOT NULL,
	Direction SMALLINT NOT NULL,
	StatTime TIMESTAMP NOT NULL,
	SourceDBConnected BIT,
	TargetDBConnected BIT,
	sourceServer VARCHAR(64),
	targetServer VARCHAR(64),
	logshipSourceTaskID INTEGER,
	logshipLogNumber INTEGER,
	logshipLogPosition INTEGER,
	logshipState INTEGER,
	logshipSequenceNumber BIGINT,
	logshipError INTEGER,
	logshipFunctionName VARCHAR(64),
	logreadSourceTaskID INTEGER,
	logreadTargetTaskID INTEGER,
	logreadLogNumber INTEGER,
	logreadLogPosition INTEGER,
	logreadState INTEGER,
	logreadSequenceNumber BIGINT,
	logreadError INTEGER,
	logreadFunctionName VARCHAR(64),
	dependencyState INTEGER,
	dependencySequenceNumber BIGINT,
	dependencyError INTEGER,
	dependencyFunctionName VARCHAR(64),
	queueStatusCount INTEGER,
	queueAnalysisCount INTEGER,
	queueDependencyCount INTEGER,
	dependencyGraphCount INTEGER,
	queueReadyCount INTEGER,
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT,
	Latency INTEGER,
	PRIMARY KEY (ID),
	UNIQUE (ReplicationPlan, Direction, StatTime)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplParallelStat ADD 
	FOREIGN KEY (ReplicationPlan)
		REFERENCES ReplicationPlan (ID);

--
-- CREATE TABLE: ReplParallelAnalizer
--
CREATE TABLE ReplParallelAnalizer
(
	replParallelStat BIGINT NOT NULL,
	seq INTEGER NOT NULL,
	state INTEGER,
	sequenceNumber BIGINT,
	error INTEGER,
	functionName VARCHAR(64)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplParallelAnalizer ADD 
	FOREIGN KEY (replParallelStat)
		REFERENCES ReplParallelStat (ID);
		
--
-- CREATE TABLE: ReplParallelApply
--
CREATE TABLE ReplParallelApply
(
	replParallelStat BIGINT NOT NULL,
	seq INTEGER NOT NULL,
	targetTaskID INTEGER,
	logNumber INTEGER,
	logPosition INTEGER,
	state INTEGER,
	files INTEGER,
	sequenceNumber BIGINT,
	error INTEGER,
	functionName VARCHAR(64),
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplParallelApply ADD 
	FOREIGN KEY (replParallelStat)
		REFERENCES ReplParallelStat (ID);
		
--
-- CREATE TABLE: ReplParallelStatHistory
--
CREATE TABLE ReplParallelStatHistory
(
	ID BIGINT,
	ReplicationPlan INTEGER NOT NULL,
	Direction SMALLINT NOT NULL,
	StatTime TIMESTAMP NOT NULL,
	SourceDBConnected BIT,
	TargetDBConnected BIT,
	sourceServer VARCHAR(64),
	targetServer VARCHAR(64),
	logshipSourceTaskID INTEGER,
	logshipLogNumber INTEGER,
	logshipLogPosition INTEGER,
	logshipState INTEGER,
	logshipSequenceNumber BIGINT,
	logshipError INTEGER,
	logshipFunctionName VARCHAR(64),
	logreadSourceTaskID INTEGER,
	logreadTargetTaskID INTEGER,
	logreadLogNumber INTEGER,
	logreadLogPosition INTEGER,
	logreadState INTEGER,
	logreadSequenceNumber BIGINT,
	logreadError INTEGER,
	logreadFunctionName VARCHAR(64),
	dependencyState INTEGER,
	dependencySequenceNumber BIGINT,
	dependencyError INTEGER,
	dependencyFunctionName VARCHAR(64),
	queueStatusCount INTEGER,
	queueAnalysisCount INTEGER,
	queueDependencyCount INTEGER,
	dependencyGraphCount INTEGER,
	queueReadyCount INTEGER,
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT,
	Latency INTEGER,
	PRIMARY KEY (ID),
	UNIQUE (ReplicationPlan, Direction, StatTime)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: ReplParallelAnalizerHistory
--
CREATE TABLE ReplParallelAnalizerHistory
(
	replParallelStat BIGINT NOT NULL,
	seq INTEGER NOT NULL,
	state INTEGER,
	sequenceNumber BIGINT,
	error INTEGER,
	functionName VARCHAR(64)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplParallelAnalizerHistory ADD 
	FOREIGN KEY (replParallelStat)
		REFERENCES ReplParallelStatHistory (ID);
		
--
-- CREATE TABLE: ReplParallelApply
--
CREATE TABLE ReplParallelApplyHistory
(
	replParallelStat BIGINT NOT NULL,
	seq INTEGER NOT NULL,
	targetTaskID INTEGER,
	logNumber INTEGER,
	logPosition INTEGER,
	state INTEGER,
	files INTEGER,
	sequenceNumber BIGINT,
	error INTEGER,
	functionName VARCHAR(64),
	AddPass BIGINT,
	DeletePass BIGINT,
	CommitPass BIGINT,
	UpdatePass BIGINT,
	CommitFail BIGINT,
	AddFail BIGINT,
	DeleteFail BIGINT,
	UpdateFail BIGINT
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplParallelApplyHistory ADD 
	FOREIGN KEY (replParallelStat)
		REFERENCES ReplParallelStatHistory (ID);

--
-- CREATE TABLE: ReplSubscription
--
CREATE TABLE ReplSubscription
(
	Subscription INTEGER NOT NULL,
	ReplicationPlan INTEGER NOT NULL,
	PRIMARY KEY (Subscription)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: ReplPlan
--
CREATE INDEX ReplPlan ON ReplSubscription
(
	ReplicationPlan
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE ReplSubscription ADD 
	FOREIGN KEY (ReplicationPlan)
		REFERENCES ReplicationPlan (ID);


--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE ReplSubscription ADD 
	FOREIGN KEY (Subscription)
		REFERENCES Subscription (ID);


--
-- CREATE TABLE: AvailGroup
--
CREATE TABLE AvailGroup
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Name VARCHAR(64) NOT NULL,
	Description VARCHAR(256),
	Status SMALLINT,
	StatusTime TIMESTAMP,
	StatusMsg VARCHAR(1024),
	StatusActionID INTEGER,
	IsDeployed BIT,
	CanFailoverToAsync BIT,
	HeartbeatFrequency INTEGER,
	HeartbeatTimeout INTEGER,
	ReplicaStatus SMALLINT,
	PRIMARY KEY (ID),
	UNIQUE (Name)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE AvailGroup ADD 
	FOREIGN KEY (Status)
		REFERENCES EngineStatus (ID);
		
--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE AvailGroup ADD 
	FOREIGN KEY (ReplicaStatus)
		REFERENCES ReplicationStatus (ID);

--
-- CREATE TABLE: AvailGroupReplica
--
CREATE TABLE AvailGroupReplica
(
	DbEngine INTEGER NOT NULL,
	AvailGroup INTEGER NOT NULL,
	Publication INTEGER,
	ReplicationPlan INTEGER,
	IsSync BIT,
	IsPrimary BIT,
	RecoverOrder INTEGER,
	TransactionID VARCHAR(32),
	PRIMARY KEY (DbEngine)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE AvailGroupReplica ADD 
	FOREIGN KEY (AvailGroup)
		REFERENCES AvailGroup (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE AvailGroupReplica ADD 
	FOREIGN KEY (DbEngine)
		REFERENCES DbEngine (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE AvailGroupReplica ADD 
	FOREIGN KEY (Publication)
		REFERENCES Publication (ID);
		
--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_4
--
ALTER TABLE AvailGroupReplica ADD 
	FOREIGN KEY (ReplicationPlan)
		REFERENCES ReplicationPlan (ID);
	
--
-- CREATE TABLE: MQTTFormat
--
CREATE TABLE MQTTFormat
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Name VARCHAR(20),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE TABLE: RetentionUnit
--
CREATE TABLE RetentionUnit
(
	ID SMALLINT NOT NULL,
	GUID BINARY(16),
	Name VARCHAR(20),
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';	
		
--
-- CREATE TABLE: MQTTPersistence
--
CREATE TABLE MQTTPersistence
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	Topic VARCHAR(512) NOT NULL,
	Status SMALLINT NOT NULL,
	StatusTime TIMESTAMP,
	StatusMsg VARCHAR(1024),
	MessageFormat SMALLINT NOT NULL,
	Threads INTEGER,
	DbEngineID INTEGER NOT NULL,
	DatabaseID INTEGER NOT NULL,
	TableName VARCHAR(64),
	Replication BIT,
	AutoTimestampEnabled BIT,
	AutoTimestampIndex BIT,
	RetentionPeriod INTEGER,
	RetentionUnit SMALLINT,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE INDEX: Name
--
CREATE INDEX Topic ON MQTTPersistence
(
	Topic
);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE MQTTPersistence ADD 
	FOREIGN KEY (DbEngineID)
		REFERENCES DbEngine (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_2
--
ALTER TABLE MQTTPersistence ADD 
	FOREIGN KEY (DatabaseID)
		REFERENCES DatabaseTable (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_3
--
ALTER TABLE MQTTPersistence ADD 
	FOREIGN KEY (Status)
		REFERENCES ReplicationStatus (ID);

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_5
--
ALTER TABLE MQTTPersistence ADD 
	FOREIGN KEY (MessageFormat)
		REFERENCES MQTTFormat (ID);
		
--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_6
--
ALTER TABLE MQTTPersistence ADD 
	FOREIGN KEY (RetentionUnit)
		REFERENCES RetentionUnit (ID);

--
-- CREATE TABLE: MQTTPersistenceField
--
CREATE TABLE MQTTPersistenceField
(
	ID INTEGER IDENTITY (0,1),
	GUID BINARY(16),
	MQTTPersistenceID INTEGER NOT NULL,
	PropertyName VARCHAR(1024) NOT NULL,
	MQTTType VARCHAR(64) NOT NULL,
	FieldName VARCHAR(64) NOT NULL,
	FieldWidth INTEGER,
	FieldScale INTEGER,
	PRIMARY KEY (ID)
) STORAGE_ATTRIBUTES 'hotalter';

--
-- CREATE FOREIGN KEY CONSTRAINT: Relation_1
--
ALTER TABLE MQTTPersistenceField ADD 
	FOREIGN KEY (MQTTPersistenceID)
		REFERENCES MQTTPersistence (ID);

--
-- METADATA POPULATION
--
insert into entitytype (id, name) values(-1, 'none');
insert into entitytype (id, name) values(0, 'site');
insert into entitytype (id, name) values(1, 'hardware');
insert into entitytype (id, name) values(2, 'opsystem');
insert into entitytype (id, name) values(3, 'volume');
insert into entitytype (id, name) values(4, 'file');
insert into entitytype (id, name) values(5, 'dbengine');
insert into entitytype (id, name) values(6, 'subscription');
insert into entitytype (id, name) values(7, 'replicationplan');

insert into actiontype (id, description) values(0, 'load_executor');
insert into actiontype (id, description) values(1, 'unload_executor');

insert into actiontype (id, description) values(100, 'fs_add');
insert into actiontype (id, description) values(101, 'fs_delete');
insert into actiontype (id, description) values(102, 'fs_change');
insert into actiontype (id, description) values(103, 'fs_scan');
insert into actiontype (id, description) values(104, 'fs_load');
insert into actiontype (id, description) values(105, 'fs_check_dbengine');
insert into actiontype (id, description) values(106, 'fs_scan_folder');
insert into actiontype (id, description) values(107, 'fs_scan_replagent');
insert into actiontype (id, description) values(108, 'fs_import_replagent');

insert into actiontype (id, description) values(200, 'deploy');
insert into actiontype (id, description) values(201, 'start');
insert into actiontype (id, description) values(202, 'stop');
insert into actiontype (id, description) values(203, 'consumer_reg');
insert into actiontype (id, description) values(204, 'consumer_reg_source');
insert into actiontype (id, description) values(205, 'consumer_unreg');
insert into actiontype (id, description) values(206, 'heartbeat');
insert into actiontype (id, description) values(207, 'start_monitor');
insert into actiontype (id, description) values(208, 'stop_monitor');
insert into actiontype (id, description) values(209, 'start_exceplog');
insert into actiontype (id, description) values(210, 'stop_exceplog');
insert into actiontype (id, description) values(211, 'start_functime');
insert into actiontype (id, description) values(212, 'stop_functime');
insert into actiontype (id, description) values(213, 'reset_functime');
insert into actiontype (id, description) values(214, 'get_functime');
insert into actiontype (id, description) values(215, 'pause');
insert into actiontype (id, description) values(216, 'resume');
insert into actiontype (id, description) values(217, 'set_trace');
insert into actiontype (id, description) values(218, 'check_failover');

insert into actiontype (id, description) values(300, 'ctree_start');
insert into actiontype (id, description) values(301, 'ctree_stop');
insert into actiontype (id, description) values(302, 'ctree_dumpfiles');
insert into actiontype (id, description) values(303, 'ctree_consumer_cancel');
insert into actiontype (id, description) values(304, 'ctree_start_monitor');
insert into actiontype (id, description) values(305, 'ctree_stop_monitor');
insert into actiontype (id, description) values(306, 'ctree_start_exceplog');
insert into actiontype (id, description) values(307, 'ctree_stop_exceplog');
insert into actiontype (id, description) values(308, 'ctree_start_functime');
insert into actiontype (id, description) values(309, 'ctree_stop_functime');
insert into actiontype (id, description) values(310, 'ctree_start_functime');
insert into actiontype (id, description) values(311, 'ctree_get_functime');
insert into actiontype (id, description) values(312, 'ctree_pause');
insert into actiontype (id, description) values(313, 'ctree_resume');
insert into actiontype (id, description) values(314, 'ctree_resync');
insert into actiontype (id, description) values(315, 'ctree_heartbeat');

insert into actiontype (id, description) values(400, 'add');
insert into actiontype (id, description) values(401, 'delete');
insert into actiontype (id, description) values(402, 'change');
insert into actiontype (id, description) values(403, 'heartbeat_target');

insert into actiontype (id, description) values(9999, 'unknown');

insert into consumer (id, name) values(-2, 'default');
insert into consumer (id, name) values(-1, 'undefined');
insert into consumer (id, name, lib) values(0, 'basic', 'ctactionexecbasic');
insert into consumer (id, name, lib) values(1, 'filesystem', 'ctactionexecfs');
insert into consumer (id, name, lib) values(2, 'repl', 'ctactionexecrepl');
insert into consumer (id, name, lib) values(3, 'repl_ctree', 'ctactionexecreplctree');
insert into consumer (id, name, lib) values(4, 'repl_nctree', 'ctactionexecreplnctree');

insert into enginestatus (id, status) values(-1, 'Invalid');
insert into enginestatus (id, status) values(0,  'Inactive');
insert into enginestatus (id, status) values(1,  'Active');
insert into enginestatus (id, status) values(2,  'Stopping');
insert into enginestatus (id, status) values(3,  'Starting');
insert into enginestatus (id, status) values(4,  'Checking');
insert into enginestatus (id, status) values(5,  'Warning');
insert into enginestatus (id, status) values(6,  'Error');
insert into enginestatus (id, status) values(7,  'Switching');
insert into enginestatus (id, status) values(8,  'Unmanaged');

insert into filetype (id, name, description) values(0, 'ctree', 'File is c-tree');
insert into filetype (id, name, description) values(1, 'ctree_corrupt', 'File is c-tree but it is corrupted');
insert into filetype (id, name, description) values(2, 'ctrep_qualif', 'File is c-tree and qualifies to replication');
insert into filetype (id, name, description) values(3, 'ctrep_qualif_rtg', 'File is c-tree RTG and qualifies to replication');
insert into filetype (id, name, description) values(4, 'ctrep_no_ifil', 'File is c-tree but doesn''t have IFIL, so doesn''t qualify for replication');
insert into filetype (id, name, description) values(5, 'ctrep_no_trnlog', 'File is c-tree but is not TRNLOG, so doesn''t qualify for replication');
insert into filetype (id, name, description) values(6, 'ctrep_no_idx', 'File is c-tree but doesn''t have an index that qualifies for replication');
insert into filetype (id, name, description) values(7, 'ctrep_no_xheader', 'File doesnt support extended header - probably created before V7');
insert into filetype (id, name, description) values(8, 'pending', 'File is pending on the schema checked - problably due to permission issue');
insert into filetype (id, name, description) values(9, 'ctreeidx', 'File is a c-tree index');
insert into filetype (id, name, description) values(10, 'ctreesuper', 'File is a c-tree super file');
insert into filetype (id, name, description) values(11, 'dir', 'Item is a directory');
insert into filetype (id, name, description) values(12, 'nctree', 'This is not a c-tree file');
insert into filetype (id, name, description) values(13, 'ctrep_fti', 'File is c-tree but has a Full Text Index, so doesn''t qualify for replication');
insert into filetype (id, name, description) values(14, 'cterror', 'File seems to be a c-tree file, but there was an error trying to open it');
insert into filetype (id, name, description) values(15, 'system', 'File is an internal system file');

insert into replicationstatus (id, name) values(0, 'Inactive');
insert into replicationstatus (id, name) values(1, 'Active');
insert into replicationstatus (id, name) values(2, 'Stopping');
insert into replicationstatus (id, name) values(3, 'Starting');
insert into replicationstatus (id, name) values(4, 'Deploying');
insert into replicationstatus (id, name) values(5, 'Paused');
insert into replicationstatus (id, name) values(6, 'Resyncing');
insert into replicationstatus (id, name) values(7, 'Archived');

insert into logtype (id, name) values(0, 'Info');
insert into logtype (id, name) values(1, 'Warning');
insert into logtype (id, name) values(2, 'Error');
insert into logtype (id, name) values(3, 'Exception');
insert into logtype (id, name) values(4, 'Trace');

insert into logagent (id, name) values(0, 'Internal');
insert into logagent (id, name) values(1, 'DBEngine');
insert into logagent (id, name) values(2, 'Replication Studio');
insert into logagent (id, name) values(3, 'Replication Agent');
insert into logagent (id, name) values(4, 'Web Server');

insert into platform (id, name, pathcasesensitive) values(0, 'winX64', 0);
insert into platform (id, name, pathcasesensitive) values(1, 'LinuxX64', 1);
insert into platform (id, name, pathcasesensitive) values(2, 'Raspbian32', 1);
insert into platform (id, name, pathcasesensitive) values(3, 'MacOSX', 1);
insert into platform (id, name, pathcasesensitive) values(4, 'Aix7', 1);

insert into layout (layoutIdx, description, zoom, panx, pany) values('1','Default',100,20,20);

insert into globals (KeyName, KeyValue) values('TRACE','YES');
insert into globals (KeyName, KeyValue) values('CTSYSTEM_FILE_LIST','ctdbdict.fsd;FAIRCOM.FCS;SEQUENCEDT.FCS;SEQUENCEIX.FCS;DFRKSTATEDT.FCS;DFRKSTATEIX.FCS;REPLSYNCDT*.FCS;REPLSYNCIX*.FCS;CTSTATUS.FCS;SNAPSHOT.FCS;sql_server.log;SYSLOGDT.FCS;SYSLOGIX.FCS;I*.FCS;D*.FCS;FREOPEN*.FCS');

insert into MQTTFormat (id, name) values(0, 'JSON');

insert into RetentionUnit (id, name) values(0, 'minute');
insert into RetentionUnit (id, name) values(1, 'hour');
insert into RetentionUnit (id, name) values(2, 'day');
insert into RetentionUnit (id, name) values(3, 'month');
insert into RetentionUnit (id, name) values(4, 'year');

COMMIT WORK;