package calendarpanel;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.FocusListener;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Calendar;
import java.util.Date;
import java.util.Vector;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JPanel;

import org.jbundle.util.calendarpanel.CalendarPane;
import org.jbundle.util.calendarpanel.CalendarPanel;
import org.jbundle.util.calendarpanel.event.MyListSelectionEvent;
import org.jbundle.util.calendarpanel.model.CalendarItem;
import org.jbundle.util.calendarpanel.util.CalendarCache;

import com.toedter.calendar.JCalendar;
import com.toedter.calendar.JMonthChooser;

public class JCalendarPanel extends JPanel {
   private static final long serialVersionUID = 1L;
   
   private JCalendar jcalendar;
   private Calendar calendar = Calendar.getInstance();
   private CalendarPanel calendarPanel;
   private Date lastDate;
   private CalendarPanelModel model = new CalendarPanelModel();
   private Vector<CalendarPanelListener> listeners = new Vector<CalendarPanelListener>();
   private Icon defaultIconStart;
   private Icon defaultIconEnd;
   private int defaultHighlightColor = 0x00e0ffff;
   private int defaultSelectColor = 0x00c0ffff;
   
   public JCalendarPanel() {
      super(new BorderLayout());
      jcalendar = new JCalendar() {
         private static final long serialVersionUID = 1L;

         public void propertyChange(java.beans.PropertyChangeEvent evt) {
            super.propertyChange(evt);
            if(listeners != null && (evt.getPropertyName().equals("date") ||
                  evt.getPropertyName().equals("day") ||
                  evt.getPropertyName().equals("month") ||
                  evt.getPropertyName().equals("year"))) {
               int oldMonth = -1;
               int oldYear = -1;
               if(lastDate != null) {
                  calendar.setTime(lastDate);
                  oldMonth = calendar.get(Calendar.MONTH);
                  oldYear = calendar.get(Calendar.YEAR);
               }
               Date newDate = jcalendar.getDate();
               calendar.setTime(newDate);
               int newMonth = calendar.get(Calendar.MONTH);
               int newYear = calendar.get(Calendar.YEAR);
               lastDate = newDate;
               if(oldMonth != newMonth || oldYear != newYear)
                  setupCalendarPanel(lastDate);
               fireDateChangedEvent(newDate);
            }
         }
      };
      calendarPanel = new CalendarPanel(model, true, null) {
         private static final long serialVersionUID = 1L;

         public void selectionChanged(MyListSelectionEvent evt) {
            super.selectionChanged(evt);
            if(evt.getType() == MyListSelectionEvent.CONTENT_CLICK) {
               CalendarCache cc = calendarPanel.getSelectedItem();
               if(cc != null) {
                  CalendarItem item = (CalendarItem) cc.getItem();
                  fireItemClickedEvent(item.getStartDate(), item.getDescription());
               }
            }
         }
         
      };
      calendarPanel.addPropertyChangeListener("date", new PropertyChangeListener() {
         
         @Override
         public void propertyChange(PropertyChangeEvent evt) {
            CalendarPane cp = calendarPanel.getComponentAtDate((Date) evt.getNewValue());
            if(cp != null)
               firePanelClickedEvent(cp.getThisDate());
         }
      });
      calendarPanel.setFocusable(false);
      JPanel calPanel = new JPanel(new BorderLayout());
      calPanel.add(jcalendar, BorderLayout.NORTH);
      calPanel.add(new JLabel(), BorderLayout.CENTER);
      add(calPanel, BorderLayout.WEST);
      add(calendarPanel, BorderLayout.CENTER);
      lastDate = jcalendar.getDate();
      setupCalendarPanel(lastDate);
   }
   
   private void setupCalendarPanel(Date d) {
      calendar.setTime(d);
      int month = calendar.get(Calendar.MONTH);
      int year = calendar.get(Calendar.YEAR);
      calendar.set(year, month, 1);
      Date startDate = calendar.getTime();
      calendar.set(year, month, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
      Date endDate = calendar.getTime();
      calendarPanel.setupCalendar(startDate, endDate, false);   
      model.setStartDate(startDate);
      model.setEndDate(endDate);
      calendarPanel.setModel(model, true);      
   }
   
   public void addCalendarPanelListener(CalendarPanelListener l) {
      if(!listeners.contains(l))
         this.listeners.addElement(l);
   }
   
   public void removeCalendarPanelListener(CalendarPanelListener l) {
      this.listeners.removeElement(l);
   }
   
   protected void fireDateChangedEvent(Date date) {
      CalendarPanelEvent evt = new CalendarPanelEvent(this, date, null, CalendarPanelEvent.DATE_CHANGED);
      for(CalendarPanelListener l : this.listeners)
         l.dateChanged(evt);
   }
   
   protected void firePanelClickedEvent(Date date) {
      CalendarPanelEvent evt = new CalendarPanelEvent(this, date, null, CalendarPanelEvent.PANEL_CLICKED);
      for(CalendarPanelListener l : this.listeners)
         l.panelClicked(evt);
   }
   
   protected void fireItemClickedEvent(Date date, String description) {
      CalendarPanelEvent evt = new CalendarPanelEvent(this, date, description, CalendarPanelEvent.ITEM_CLICKED);
      for(CalendarPanelListener l : this.listeners)
         l.itemClicked(evt);
   }
   
   public void setFont(Font f) {
      super.setFont(f);
      if(jcalendar != null) {
         jcalendar.setFont(f);
         JMonthChooser mc = jcalendar.getMonthChooser();
         if(mc != null) {
            mc.setPreferredSize(
                  new Dimension(mc.getPreferredSize().width, 
                        getFontMetrics(f).getHeight() + 10));
         } 
         calendarPanel.setFont(f);
      }
   }
   
   public void setDate(Date d) {
      jcalendar.setDate(d);
   }
   
   public Date getDate() {
      return jcalendar.getDate();
   }
   
   public void addItem(Date startDate, String description) {
      model.addItem(new CalendarPanelItem(startDate, 
                                          null, 
                                          description, 
                                          null, 
                                          getDefaultIconStart(), 
                                          getDefaultIconEnd(), 
                                          getDefaultHighlightColor(), 
                                          getDefaultSelectColor()));
      model.fireTableRowsInserted(model.getRowCount() - 1, model.getRowCount() - 1);
   }
   
   public void addItem(Date startDate, Date endDate, String description,
         String mealDescription, Icon iconStart, Icon iconEnd,
         int highlightColor, int selectColor) {
      model.addItem(new CalendarPanelItem(startDate, 
                                          endDate, 
                                          description,
                                          mealDescription, 
                                          iconStart, 
                                          iconEnd, 
                                          highlightColor, 
                                          selectColor));
      model.fireTableRowsInserted(model.getRowCount() - 1, model.getRowCount() - 1);
   }
   
   public CalendarPanelItem[] getItems(Date date) {
      return model.getItems(date);
   }
   
   public CalendarPanelItem[] getItems() {
      return model.getItems();
   }
   
   public void removeAllItems() {
      model.free();
   }
   
   public CalendarPanelItem[] removeItems(Date date) {
      CalendarPanelItem[] Return = model.getItems(date);
      for(CalendarPanelItem it : Return)
         model.removeItem(it);
      return Return;
   }
   
   public void removeItem(Date date, String description) {
      model.removeItem(new CalendarPanelItem(date, description));
   }

   public Icon getDefaultIconStart() {
      return defaultIconStart;
   }

   public void setDefaultIconStart(Icon defaultIconStart) {
      this.defaultIconStart = defaultIconStart;
   }

   public Icon getDefaultIconEnd() {
      return defaultIconEnd;
   }

   public void setDefaultIconEnd(Icon defaultIconEnd) {
      this.defaultIconEnd = defaultIconEnd;
   }

   public int getDefaultHighlightColor() {
      return defaultHighlightColor;
   }

   public void setDefaultHighlightColor(int defaultHighlightColor) {
      this.defaultHighlightColor = defaultHighlightColor;
   }

   public int getDefaultSelectColor() {
      return defaultSelectColor;
   }

   public void setDefaultSelectColor(int defaultSelectColor) {
      this.defaultSelectColor = defaultSelectColor;
   }
   
   public void requestFocus() {
      jcalendar.requestFocus();
   }
   
   public boolean requestFocusInWindow() {
      return jcalendar.requestFocusInWindow();
   }
   
   public void addKeyListener(KeyListener kl) {
      jcalendar.addKeyListener(kl);
   }
   
   public void removeKeyListener(KeyListener kl) {
      jcalendar.removeKeyListener(kl);
   }
   
   public void addFocusListener(FocusListener fl) {
      jcalendar.addFocusListener(fl);
   }
   
   public void removeFocusListener(FocusListener fl) {
      jcalendar.removeFocusListener(fl);
   }
   
   public void addMouseListener(MouseListener ml) {
      jcalendar.addMouseListener(ml);
      calendarPanel.addMouseListener(ml);
   }
   
   public void removeMouseListener(MouseListener ml) {
      jcalendar.removeMouseListener(ml);
      calendarPanel.removeMouseListener(ml);
   }
   
   public void addMouseMotionListener(MouseMotionListener ml) {
      jcalendar.addMouseMotionListener(ml);
      calendarPanel.addMouseMotionListener(ml);
   }
   
   public void removeMouseMotionListener(MouseMotionListener ml) {
      jcalendar.removeMouseMotionListener(ml);
      calendarPanel.removeMouseMotionListener(ml);
   }
}
